local awful = require('awful')
local gears = require('gears')
local ruled = require('ruled')
local beautiful = require('beautiful')
local dpi = beautiful.xresources.apply_dpi
local client_keys = require('configuration.client.keys')
local client_buttons = require('configuration.client.buttons')

ruled.client.connect_signal(
	'request::rules',
	function()
		-- All clients will match this rule.
		ruled.client.append_rule {
			id = 'global',
			rule = {},
			except_any = { type = "desktop" },
			properties = {
				focus = awful.client.focus.filter,
				raise = true,
				floating = false,
				maximized = false,
				above = false,
				below = false,
				ontop = false,
				sticky = false,
				maximized_horizontal = false,
				maximized_vertical = false,
				keys = client_keys,
				buttons = client_buttons,
				placement = awful.placement.no_overlap + awful.placement.no_offscreen
			}
		}

		ruled.client.append_rule {
			id = 'round_clients',
			rule_any = {
				type = {
					'normal',
					'dialog'
				}
			},
			properties = {
				round_corners = true,
				shape = beautiful.client_shape_rounded
			}
		}

		-- Titlebar rules
		ruled.client.append_rule {
			id 		= 'titlebars',
			rule_any = {
				type = {
					'normal',
					'dialog'
				}
			},
			properties = {
				titlebars_enabled = true
			}
		}

		-- Dialogs
		ruled.client.append_rule {
			id = 'dialog',
			rule_any = {
				type  = {'dialog'},
				class = {'Wicd-client.py', 'calendar.google.com'}
			},
			properties = {
				titlebars_enabled = true,
				floating = false,
				above = true,
				skip_decoration = false,
				placement = awful.placement.centered
			}
		}
		
		ruled.client.append_rule {
			id = 'on_screen_keyboard',
			rule_any = {
				class = {
					'Onboard'
				}
			},
			properties = {
				focusable = false,
				ontop = true,
				skip_decoration = true,
				floating = true
			}
		}
		
		ruled.client.append_rule {
			id = 'picture-in-picture',
			rule_any = {
				instance = {
					'Toolkit'
				},
				sticky = true
			},
			properties = {
				sticky = true,
				ontop = true
			}
		}

		-- Modals
		ruled.client.append_rule {
			id = 'modal',
			rule_any = {
				type = {'modal'}
			},
			properties = {
				titlebars_enabled = true,
				floating = true,
				above = true,
				titlebars_enabled = false,
				skip_decoration = true,
				placement = awful.placement.centered
			}
		}

		-- Utilities
		ruled.client.append_rule {
			id = 'utility',
			rule_any = {
				type = {'utility'}
			},
			properties = {
				titlebars_enabled = false,
				floating = true,
				skip_decoration = true,
				placement = awful.placement.centered
			}
		}

		-- Splash
		ruled.client.append_rule {
			id = 'splash',
			rule_any = {
				type = {'splash'},
				name = {"Discord Updater"}
			},
			properties = {
				titlebars_enabled = false,
				round_corners = true,
				floating = true,
				above = true,
				skip_decoration = true,
				placement = awful.placement.centered
			}
		}
		
		-- Desktop-class clients like NebiDE-Wallpaper-Engine and conky
		ruled.client.append_rule {
			id = 'desktop',
			rule_any = {
				type = {'desktop'}
			},
			properties = {
				titlebars_enabled = false,
				round_corners = false,
				floating = true,
				skip_taskbar = true,
				sticky = true,
				below = true,
				skip_decoration = true
			}
		}
		
		ruled.client.append_rule {
			id = 'NebiDE-WE',
			rule_any = {
				class = {'NebiDE_WE'}
			},
			properties = {
			    moveable = false,
			    floating = true,
			    maximized = false,
			    resizeable = false,
			    ewmh = false
			}
		}
		
		ruled.client.append_rule {
			id = 'NebiDE_applauncher',
			rule_any = {
				class = {'NebiDE_launcher'}
			},
			properties = {
				titlebars_enabled = false,
				floating = true,
				above = true,
            			placement = function(c)
				    return awful.placement.left(c, { margins = {
				    	top = dpi(10) + c.screen.top_panel.height,
				    	left = dpi(10),
				    	bottom = dpi(10)
				    } })
				end,
            			margins = dpi(10),
				skip_decoration = true
			}
		}
				
		ruled.client.append_rule {
			id = 'jetbrains',
			rule_any = {
				name = {"win0", "Emulator"}
			},
			properties = {
				titlebars_enabled = false,
				round_corners = false,
				floating = true,
				above = false,
				skip_decoration = true,
				placement = awful.placement.centered
			}
		}
		
		ruled.client.append_rule {
			id = 'resolve',
			rule_any = {
				class = {"resolve"}
			},
			properties = {
				border_width = dpi(1),
				callback = function(c)
					c.force_titlebars_enabled = true
					c:emit_signal("request::titlebars")
				end
			}
		}
		
		ruled.client.append_rule {
			id = 'NebiDE_right_info',
			rule_any = {
				name = {"NebiDE_right_off"}
			},
			properties = {
				 focusable  = false
			}
		}
		
		-- Floating
		ruled.client.append_rule {
			id       = 'floating',
			rule_any = {
				instance    = {
					'file_progress',
					'Popup',
					'nm-connection-editor',
				},
				class = {
					'scrcpy',
					'Mugshot',
					'Pulseeffects'
				},
				role    = {
					'AlarmWindow',
					'ConfigManager',
					'pop-up'
				}
			},
			properties = {
				titlebars_enabled = true,
				skip_decoration = true,
				ontop = true,
				floating = true,
				focus = awful.client.focus.filter,
				raise = true,
				keys = client_keys,
				buttons = client_buttons,
				placement = awful.placement.centered
			}
		}
	end
)

-- Normally we'd do this with a rule, but some program like spotify doesn't set its class or name
-- until after it starts up, so we need to catch that signal.
client.connect_signal(
	'property::class',
	function(c)
		if c.class == 'Spotify' then
			local window_mode = false

			-- Check if fullscreen or window mode
			if c.fullscreen then
				window_mode = false
				c.fullscreen = false
			else
				window_mode = true
			end

			-- Check if Spotify is already open
			local app = function (c)
				return ruled.client.match(c, {class = 'Spotify'})
			end

			local app_count = 0
			for c in awful.client.iterate(app) do
				app_count = app_count + 1
			end

			-- If Spotify is already open, don't open a new instance
			if app_count > 1 then
				c:kill()
				-- Switch to previous instance
				for c in awful.client.iterate(app) do
					c:jump_to(false)
				end
			else
				-- Move the instance to specified tag on this screen
				local t = awful.tag.find_by_name(awful.screen.focused(), '5')
				c:move_to_tag(t)
				t:view_only()

				-- Fullscreen mode if not window mode
				if not window_mode then
					c.fullscreen = true
				else
					c.floating = true
					awful.placement.centered(c, {honor_workarea = true})
				end
			end
		end
	end
)
